document.addEventListener('DOMContentLoaded', function() {
    // DOM Elements
    const sidebar = document.getElementById('sidebar');
    const sidebarToggle = document.getElementById('sidebar-toggle');
    const mobileSidebarToggle = document.getElementById('mobile-sidebar-toggle');
    const userMenuButton = document.getElementById('user-menu-button');
    const userMenu = document.getElementById('user-menu');
    const contentSections = document.querySelectorAll('.content-section');
    const navLinks = document.querySelectorAll('nav a');
    const generateQRBtn = document.getElementById('generate-qr-btn');
    const qrContainer = document.getElementById('qr-container');
    const qrContent = document.getElementById('qr-content');
    const qrSpinner = document.getElementById('qr-spinner');
    const qrExpiry = document.getElementById('qr-expiry');
    const statusIndicator = document.getElementById('status-indicator');
    const statusText = document.getElementById('status-text');
    const usernameDisplay = document.getElementById('username');
    const connectionStatus = document.getElementById('connection-status');
    
    // Socket.IO connection
    const socket = io('http://161.248.162.167:3000');
    let userId = null;
  
    // Initialize the dashboard
    initDashboard();
  
    // Functions
    function initDashboard() {
      // Load user data
      loadUser();
      
      // Setup event listeners
      setupEventListeners();
      
      // Initialize charts
      initCharts();
    }
  
    function loadUser() {
      fetch('/api/user')
        .then(res => {
          if (res.status === 401) {
            window.location.href = 'login.html';
            return Promise.reject('Unauthorized');
          }
          return res.json();
        })
        .then(data => {
          if (data.error) {
            window.location.href = 'login.html';
            return;
          }
          userId = data.id;
          usernameDisplay.textContent = data.username;
          document.getElementById('user-username').textContent = data.username;
          
          // Join user room
          socket.emit('join-user', userId);
          
          // Load initial data
          loadInitialData();
        })
        .catch(err => {
          console.error('Failed to load user:', err);
          window.location.href = 'login.html';
        });
    }
  
    function loadInitialData() {
      loadStats();
      loadMessages();
      loadConnectionStatus();
    }
  
    function loadStats() {
      fetch('/api/stats')
        .then(res => res.json())
        .then(data => {
          document.getElementById('sent-count').textContent = data.sent || 0;
          document.getElementById('last-active').textContent = data.last_activity ? 
            new Date(data.last_activity).toLocaleString() : 'Never';
        })
        .catch(err => {
          console.error('Failed to load stats:', err);
          showNotification('Failed to load statistics', 'error');
        });
    }
  
    function loadMessages() {
      fetch('/api/messages')
        .then(res => res.json())
        .then(messages => {
          const chatContainer = document.getElementById('chat-container');
          chatContainer.innerHTML = '';
          
          if (messages.length === 0) {
            chatContainer.innerHTML = '<div class="text-center text-gray-500 py-8">No recent messages</div>';
            return;
          }
          
          messages.reverse().forEach(msg => {
            const messageDiv = document.createElement('div');
            messageDiv.className = `message ${msg.direction}`;
            messageDiv.innerHTML = `
              <div class="message-content">${msg.message}</div>
              <div class="timestamp">${new Date(msg.created_at).toLocaleString()}</div>
            `;
            chatContainer.appendChild(messageDiv);
          });
        })
        .catch(err => {
          console.error('Failed to load messages:', err);
          showNotification('Failed to load messages', 'error');
        });
    }
  
    function loadConnectionStatus() {
      fetch('/api/connection-status')
        .then(res => res.json())
        .then(data => {
          updateStatus(data.status);
        })
        .catch(err => {
          console.error('Failed to load connection status:', err);
        });
    }
  
    function updateStatus(status) {
      // Update status indicator
      statusIndicator.className = `status-indicator status-${status}`;
      statusText.textContent = status.charAt(0).toUpperCase() + status.slice(1);
      statusText.className = `text-sm ${status === 'connected' ? 'text-emerald-600' : 
        status === 'connecting' ? 'text-amber-600' : 'text-red-600'}`;
      
      // Update topbar connection status
      const connectionDot = connectionStatus.querySelector('span');
      connectionDot.className = `w-3 h-3 rounded-full mr-2 ${status === 'connected' ? 'bg-emerald-500' : 
        status === 'connecting' ? 'bg-amber-500' : 'bg-red-500'}`;
      connectionStatus.querySelector('span:last-child').textContent = 
        status.charAt(0).toUpperCase() + status.slice(1);
      
      // Update QR login section if needed
      if (status === 'connected') {
        document.getElementById('qr-connected-info').classList.remove('hidden');
        document.getElementById('qr-login-content').classList.add('hidden');
        document.getElementById('qr-login-spinner').classList.add('hidden');
      }
    }
  
    function setupEventListeners() {
      // Sidebar toggle for mobile
      mobileSidebarToggle.addEventListener('click', () => {
        sidebar.classList.toggle('-translate-x-full');
      });
  
      sidebarToggle.addEventListener('click', () => {
        sidebar.classList.toggle('-translate-x-full');
      });
  
      // User menu toggle
      userMenuButton.addEventListener('click', () => {
        userMenu.classList.toggle('hidden');
      });
  
      // Close user menu when clicking outside
      document.addEventListener('click', (e) => {
        if (!userMenuButton.contains(e.target) && !userMenu.contains(e.target)) {
          userMenu.classList.add('hidden');
        }
      });
  
      // Navigation links
      navLinks.forEach(link => {
        link.addEventListener('click', (e) => {
          e.preventDefault();
          
          // Update active state
          navLinks.forEach(l => l.classList.remove('text-emerald-700', 'font-medium', 'bg-emerald-50'));
          link.classList.add('text-emerald-700', 'font-medium', 'bg-emerald-50');
          
          // Show corresponding section
          const sectionId = link.getAttribute('href').substring(1);
          contentSections.forEach(section => {
            section.classList.add('hidden');
            section.classList.remove('active');
          });
          document.getElementById(sectionId).classList.remove('hidden');
          document.getElementById(sectionId).classList.add('active');
          
          // Load section-specific data
          if (sectionId === 'analytics') {
            loadAnalyticsData();
          } else if (sectionId === 'qr-login') {
            loadQRLoginData();
          }
        });
      });
  
      // Generate QR button
      generateQRBtn.addEventListener('click', () => {
        generateQRCode();
      });
  
      // Logout
      document.querySelectorAll('[href="#logout"]').forEach(link => {
        link.addEventListener('click', (e) => {
          e.preventDefault();
          logout();
        });
      });
    }
  
    function generateQRCode() {
      if (!userId) return;
  
      qrSpinner.classList.remove('hidden');
      qrContent.innerHTML = '';
      qrExpiry.textContent = '';
      qrContainer.classList.remove('hidden');
  
      fetch('/api/generate-qr', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        }
      })
        .then(res => res.json())
        .then(data => {
          if (data.error) {
            showNotification(data.error, 'error');
            qrContainer.classList.add('hidden');
            qrSpinner.classList.add('hidden');
          }
        })
        .catch(err => {
          console.error('Failed to generate QR code:', err);
          showNotification('Failed to generate QR code', 'error');
          qrContainer.classList.add('hidden');
          qrSpinner.classList.add('hidden');
        });
    }
  
    function loadQRLoginData() {
      const qrLoginSpinner = document.getElementById('qr-login-spinner');
      const qrLoginContent = document.getElementById('qr-login-content');
      const qrConnectedInfo = document.getElementById('qr-connected-info');
      
      qrLoginSpinner.classList.remove('hidden');
      qrLoginContent.classList.add('hidden');
      qrConnectedInfo.classList.add('hidden');
      
      fetch('/api/connection-status')
        .then(res => res.json())
        .then(data => {
          if (data.status === 'connected') {
            qrLoginSpinner.classList.add('hidden');
            qrConnectedInfo.classList.remove('hidden');
            // Load connected device info
            loadConnectedDeviceInfo();
          } else {
            generateQRForLogin();
          }
        })
        .catch(err => {
          console.error('Failed to check connection status:', err);
          generateQRForLogin();
        });
    }
  
    function generateQRForLogin() {
      const qrLoginSpinner = document.getElementById('qr-login-spinner');
      const qrLoginContent = document.getElementById('qr-login-content');
      const qrLoginImage = document.getElementById('qr-login-image');
      const qrExpiryTime = document.getElementById('qr-expiry-time');
      
      qrLoginSpinner.classList.remove('hidden');
      qrLoginContent.classList.add('hidden');
      
      fetch('/api/generate-qr', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json'
        }
      })
        .then(res => res.json())
        .then(data => {
          if (data.error) {
            showNotification(data.error, 'error');
            return;
          }
          
          // The QR code will be received via socket.io
        })
        .catch(err => {
          console.error('Failed to generate QR code:', err);
          showNotification('Failed to generate QR code', 'error');
        });
    }
  
    function loadConnectedDeviceInfo() {
      // In a real implementation, this would fetch actual device info
      document.getElementById('connected-number').textContent = '+1234567890';
      document.getElementById('connected-device').textContent = 'Android';
      document.getElementById('connection-time').textContent = 'Just now';
    }
  
    function loadAnalyticsData() {
      // This would load all analytics charts and data
      console.log('Loading analytics data...');
    }
  
    function initCharts() {
      // Initialize any charts needed for the dashboard
    }
  
    function logout() {
      fetch('/api/logout', {
        method: 'POST'
      })
        .then(res => res.json())
        .then(data => {
          if (data.success) {
            window.location.href = 'login.html';
          }
        })
        .catch(err => {
          console.error('Logout failed:', err);
          showNotification('Failed to logout', 'error');
        });
    }
  
    function showNotification(message, type = 'info', duration = 5000) {
      const container = document.getElementById('notification-container');
      const notification = document.createElement('div');
      
      notification.className = `notification notification-${type}`;
      notification.innerHTML = `
        <span>${message}</span>
        <button class="notification-close"><i class="fas fa-times"></i></button>
      `;
      
      container.appendChild(notification);
      
      const closeBtn = notification.querySelector('.notification-close');
      closeBtn.addEventListener('click', () => {
        notification.classList.add('hide');
        setTimeout(() => notification.remove(), 300);
      });
      
      if (duration) {
        setTimeout(() => {
          notification.classList.add('hide');
          setTimeout(() => notification.remove(), 300);
        }, duration);
      }
    }
  
    // Socket.IO listeners
    socket.on(`connection-status-${userId}`, ({ status }) => {
      updateStatus(status);
    });
  
    socket.on(`qr-${userId}`, ({ qr, text, expiresAt }) => {
      // For home section
      qrSpinner.classList.add('hidden');
      qrContent.innerHTML = `<img src="${qr}" class="qr-code" alt="WhatsApp QR Code">`;
      qrExpiry.textContent = `Expires at: ${new Date(expiresAt).toLocaleTimeString()}`;
      
      // For QR login section
      const qrLoginSpinner = document.getElementById('qr-login-spinner');
      const qrLoginContent = document.getElementById('qr-login-content');
      const qrLoginImage = document.getElementById('qr-login-image');
      const qrExpiryTime = document.getElementById('qr-expiry-time');
      
      qrLoginSpinner.classList.add('hidden');
      qrLoginContent.classList.remove('hidden');
      qrLoginImage.innerHTML = `<img src="${qr}" class="qr-code mx-auto" alt="WhatsApp QR Code">`;
      
      // Update expiry time countdown
      const expiryTime = new Date(expiresAt).getTime();
      const updateExpiryTime = () => {
        const now = new Date().getTime();
        const remaining = Math.max(0, Math.floor((expiryTime - now) / 1000));
        qrExpiryTime.textContent = remaining;
        
        if (remaining > 0) {
          setTimeout(updateExpiryTime, 1000);
        } else {
          qrLoginContent.classList.add('hidden');
          qrLoginSpinner.classList.remove('hidden');
        }
      };
      
      updateExpiryTime();
      
      showNotification('QR code generated. Scan with WhatsApp to connect', 'info');
    });
  
    socket.on(`error-${userId}`, (error) => {
      console.error('Error:', error);
      showNotification(`Error: ${error.message}`, 'error');
    });
  
    socket.on(`new_message-${userId}`, ({ from, message, reply }) => {
      loadStats();
      loadMessages();
      showNotification(`New message from ${from.split('@')[0]}`, 'info');
    });
  
    socket.on(`settings-updated-${userId}`, () => {
      showNotification('Settings updated successfully', 'success');
    });
  
    socket.on(`auto-replies-updated-${userId}`, () => {
      showNotification('Auto-replies updated', 'success');
    });
  });