const { makeWASocket, useMultiFileAuthState, DisconnectReason } = require('@whiskeysockets/baileys');
const express = require('express');
const socketIO = require('socket.io');
const http = require('http');
const path = require('path');
const mysql = require('mysql2/promise');
const pino = require('pino');
const QRCode = require('qrcode');
const moment = require('moment');
const nodemailer = require('nodemailer');
const session = require('express-session');
const MySQLStore = require('express-mysql-session')(session);
const bcrypt = require('bcryptjs');
const rateLimit = require('express-rate-limit');
const cluster = require('cluster');
const os = require('os');

// Number of CPU cores for clustering
const numCPUs = os.cpus().length;

// Database configuration with increased connection limit
const pool = mysql.createPool({
  host: 'localhost',
  user: 'root',
  password: '',
  database: 'whatsapp_bot',
  waitForConnections: true,
  connectionLimit: 50, // Increased for 100 users
  queueLimit: 0
});

// Session store
const sessionStore = new MySQLStore({
  host: 'localhost',
  user: 'root',
  password: '',
  database: 'whatsapp_bot'
});

// Express setup
const app = express();
const server = http.createServer(app);
const io = socketIO(server, {
  cors: {
    origin: [
      "http://161.248.162.167:3000",
      "http://161.248.162.167",
      "http://localhost:3000"
    ],
    methods: ["GET", "POST"],
    credentials: true
  }
});

// Session middleware
app.use(session({
  secret: 'b9346f58ebc996608882f36b36653d5a6f78c17efea736964687134dcfde145e',
  store: sessionStore,
  resave: false,
  saveUninitialized: false,
  cookie: {
    maxAge: 24 * 60 * 60 * 1000,
    httpOnly: true,
    secure: false
  }
}));

// Rate limiter for API endpoints
const apiLimiter = rateLimit({
  windowMs: 15 * 60 * 1000, // 15 minutes
  max: 100, // Limit each IP to 100 requests per windowMs
  message: 'Too many requests, please try again later.'
});
app.use('/api/', apiLimiter);

// SMTP configuration
const transporter = nodemailer.createTransport({
  host: 'mail.joycloud.services',
  port: 465,
  secure: true,
  auth: {
    user: 'striker@joycloud.services',
    pass: 'teamstriker990089aR@'
  },
  tls: { rejectUnauthorized: false },
  connectionTimeout: 10000,
  greetingTimeout: 10000
});

// Verify SMTP connection
transporter.verify((error, success) => {
  if (error) {
    console.error('SMTP verification error:', error);
  } else {
    console.log('SMTP server is ready to send emails');
  }
});

// Middleware
app.use(express.json());
app.use(express.static(path.join(__dirname, 'public')));

// Authentication middleware
const isAuthenticated = (req, res, next) => {
  if (req.session.userId) {
    return next();
  }
  res.redirect('/login');
};

// Generate verification code
const generateVerificationCode = () => {
  return Math.floor(100000 + Math.random() * 900000).toString();
};

// Map to store WhatsApp connections and statuses per user
const userConnections = new Map(); // { userId: { sock, status, isGeneratingQR } }
const afkCooldowns = new Map(); // { userId: { sender: timestamp } }

// Log error to database
async function logError(userId, type, message) {
  try {
    const [connections] = await pool.query('SELECT id FROM connections WHERE user_id = ? LIMIT 1', [userId]);
    const connectionId = connections[0]?.id || userId;
    await pool.query(
      'INSERT INTO error_logs (user_id, connection_id, type, message) VALUES (?, ?, ?, ?)',
      [userId, connectionId, type, message]
    );
    io.to(`user-${userId}`).emit(`error-${userId}`, { type, message, timestamp: new Date().toISOString() });
  } catch (error) {
    console.error('Failed to log error:', error);
  }
}

// Initialize WhatsApp bot for a specific user
async function initWhatsAppBot(userId) {
  try {
    const { state, saveCreds } = await useMultiFileAuthState(`auth_info_${userId}`);
    const logger = pino({ level: 'silent' });

    const sock = makeWASocket({
      auth: state,
      printQRInTerminal: false,
      logger: logger
    });

    // Store connection details
    userConnections.set(userId, {
      sock,
      status: 'disconnected',
      isGeneratingQR: false
    });

    sock.ev.on('creds.update', saveCreds);

    sock.ev.on('connection.update', async (update) => {
      const { connection, lastDisconnect, qr } = update;

      if (qr) {
        console.log(`QR code generated for user: ${userId}`);
        userConnections.get(userId).status = 'connecting';
        userConnections.get(userId).isGeneratingQR = true;
        io.to(`user-${userId}`).emit(`connection-status-${userId}`, { status: 'connecting' });

        try {
          const qrImage = await QRCode.toDataURL(qr, { errorCorrectionLevel: 'H' });
          io.to(`user-${userId}`).emit(`qr-${userId}`, {
            qr: qrImage,
            text: 'Scan this QR code with WhatsApp',
            expiresAt: Date.now() + 60000
          });
          await pool.query('UPDATE connections SET status = ? WHERE user_id = ?', ['connecting', userId]);
        } catch (error) {
          console.error(`Failed to generate QR code for user ${userId}:`, error);
          userConnections.get(userId).status = 'disconnected';
          io.to(`user-${userId}`).emit(`connection-status-${userId}`, { status: 'disconnected' });
          await logError(userId, 'QR Generation', 'Failed to generate QR code image');
        }
      }

      if (connection === 'close') {
        const shouldReconnect = lastDisconnect?.error?.output?.statusCode !== DisconnectReason.loggedOut;
        userConnections.get(userId).status = shouldReconnect ? 'disconnected' : 'logged-out';
        userConnections.get(userId).isGeneratingQR = false;
        io.to(`user-${userId}`).emit(`connection-status-${userId}`, { status: userConnections.get(userId).status });
        await pool.query('UPDATE connections SET status = ? WHERE user_id = ?', [userConnections.get(userId).status, userId]);

        if (shouldReconnect) {
          console.log(`Connection closed, reconnecting for user: ${userId}`);
          setTimeout(() => initWhatsAppBot(userId), 3000);
        } else {
          userConnections.delete(userId);
        }
      } else if (connection === 'open') {
        console.log(`Connected to WhatsApp for user: ${userId}`);
        userConnections.get(userId).status = 'connected';
        userConnections.get(userId).isGeneratingQR = false;
        io.to(`user-${userId}`).emit(`connection-status-${userId}`, { status: 'connected' });
        await pool.query('UPDATE connections SET status = ? WHERE user_id = ?', ['connected', userId]);
      }
    });

    sock.ev.on('messages.upsert', async ({ messages }) => {
      try {
        const msg = messages[0];
        if (!msg.message || msg.key.fromMe) {
          console.log(`Skipping message for user ${userId}: no content or sent by bot`);
          return;
        }

        const sender = msg.key.remoteJid;
        let message = '';

        // Handle different message types
        if (msg.message.conversation) {
          message = msg.message.conversation;
        } else if (msg.message.extendedTextMessage?.text) {
          message = msg.message.extendedTextMessage.text;
        } else if (msg.message.imageMessage?.caption) {
          message = msg.message.imageMessage.caption;
        } else if (msg.message.videoMessage?.caption) {
          message = msg.message.videoMessage.caption;
        } else if (msg.message.documentMessage?.caption) {
          message = msg.message.documentMessage.caption;
        } else if (msg.message.audioMessage) {
          message = '[Audio message]';
        } else {
          console.log(`Unsupported message type for user ${userId}:`, msg.message);
          return;
        }

        console.log(`Processing message for user ${userId}:`, { sender, message });

        const isGroup = sender.includes('@g.us');
        const [connections] = await pool.query('SELECT id FROM connections WHERE user_id = ? LIMIT 1', [userId]);
        if (connections.length === 0) {
          console.error(`No connection found for user: ${userId}`);
          await logError(userId, 'Message Processing', 'No connection found');
          return;
        }
        const connectionId = connections[0].id;

        // Log incoming message
        await pool.query(
          'INSERT INTO messages (user_id, connection_id, sender, message, direction) VALUES (?, ?, ?, ?, ?)',
          [userId, connectionId, sender, message, 'incoming']
        );

        // Fetch settings
        let [settings] = await pool.query('SELECT * FROM settings WHERE user_id = ?', [userId]);
        let config = settings[0];
        if (!config) {
          console.log(`No settings found, inserting defaults for user: ${userId}`);
          await pool.query(
            'INSERT INTO settings (id, user_id, connection_id, auto_reply_enabled, auto_reply_groups, afk_enabled, afk_message) VALUES (?, ?, ?, ?, ?, ?, ?)',
            [userId, userId, connectionId, false, false, false, 'I am currently unavailable. Please leave a message.']
          );
          config = {
            auto_reply_enabled: false,
            auto_reply_groups: false,
            afk_enabled: false,
            afk_message: 'I am currently unavailable. Please leave a message.'
          };
        }

        let reply = null;

        // Initialize AFK cooldowns for user if not exists
        if (!afkCooldowns.has(userId)) {
          afkCooldowns.set(userId, new Map());
        }
        const userAfkCooldowns = afkCooldowns.get(userId);

        // AFK logic with cooldown
        if (config.afk_enabled && !isGroup) {
          const lastReplyTime = userAfkCooldowns.get(sender) || 0;
          const now = Date.now();
          if (now - lastReplyTime > 5 * 60 * 1000) {
            reply = config.afk_message || 'I am currently unavailable. Please leave a message.';
            userAfkCooldowns.set(sender, now);
            console.log(`Sending AFK reply to ${sender} for user ${userId}`);
          } else {
            console.log(`AFK reply skipped due to cooldown for ${sender} for user ${userId}`);
          }
        }
        // Auto Reply logic
        else if (config.auto_reply_enabled && (!isGroup || config.auto_reply_groups)) {
          const [autoReplies] = await pool.query('SELECT * FROM auto_replies WHERE user_id = ?', [userId]);
          const messageLower = message.toLowerCase();
          const matchedReply = autoReplies.find(reply => {
            const keyword = reply.keyword.toLowerCase();
            const regex = new RegExp(`\\b${keyword}\\b`, 'i');
            return regex.test(messageLower);
          });
          if (matchedReply) {
            reply = matchedReply.response;
            console.log(`Matched auto-reply for user ${userId}:`, { keyword: matchedReply.keyword, response: reply });
          } else {
            console.log(`No auto-reply matched for message for user ${userId}:`, message);
          }
        }

        if (reply && userConnections.get(userId)?.sock) {
          try {
            await userConnections.get(userId).sock.sendMessage(sender, { text: reply });
            await pool.query(
              'INSERT INTO messages (user_id, connection_id, sender, message, direction) VALUES (?, ?, ?, ?, ?)',
              [userId, connectionId, sender, reply, 'outgoing']
            );
            console.log(`Reply sent for user ${userId}:`, { sender, reply });
          } catch (sendError) {
            console.error(`Failed to send reply for user ${userId}:`, sendError);
            await logError(userId, 'Message Sending', sendError.message);
          }
        }

        io.to(`user-${userId}`).emit(`new_message-${userId}`, {
          from: sender,
          message: message,
          reply: reply !== null,
          timestamp: new Date().toISOString()
        });
      } catch (error) {
        console.error(`Message processing error for user ${userId}:`, error.stack);
        await logError(userId, 'Message Processing', error.message);
      }
    });

    return sock;
  } catch (error) {
    console.error(`WhatsApp initialization error for user ${userId}:`, error);
    userConnections.get(userId).status = 'disconnected';
    io.to(`user-${userId}`).emit(`connection-status-${userId}`, { status: 'disconnected' });
    await logError(userId, 'Initialization', error.message);
    userConnections.delete(userId);
    throw error;
  }
}

// Auth endpoints
app.post('/api/register', async (req, res) => {
  const { email, password, username } = req.body;
  console.log('Registration attempt:', { email, username });
  try {
    if (!email || !password || !username) {
      return res.status(400).json({ error: 'All fields are required' });
    }
    if (!/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email)) {
      return res.status(400).json({ error: 'Invalid email format' });
    }
    if (password.length < 6) {
      return res.status(400).json({ error: 'Password must be at least 6 characters' });
    }

    const [existingUsers] = await pool.query('SELECT * FROM users WHERE email = ?', [email]);
    if (existingUsers.length > 0) {
      return res.status(400).json({ error: 'Email already exists' });
    }

    const verificationCode = generateVerificationCode();
    const hashedPassword = await bcrypt.hash(password, 10);

    await pool.query(
      'INSERT INTO users (username, email, password, verification_code, is_verified) VALUES (?, ?, ?, ?, ?)',
      [username, email, hashedPassword, verificationCode, false]
    );

    try {
      await transporter.sendMail({
        from: '"WhatsApp Bot" <striker@joycloud.services>',
        to: email,
        subject: 'Verify Your Account',
        html: `
          <h2>WhatsApp Bot Verification</h2>
          <p>Hello ${username},</p>
          <p>Your verification code is: <strong>${verificationCode}</strong></p>
          <p>Please enter this code on the verification page to activate your account.</p>
        `
      });
      console.log(`Verification email sent to ${email}`);
    } catch (emailError) {
      console.error('Email sending error:', emailError);
      await pool.query('DELETE FROM users WHERE email = ?', [email]);
      return res.status(500).json({ error: 'Failed to send verification email' });
    }

    req.session.registrationEmail = email;
    res.json({ success: true, message: 'Verification code sent to your email' });
  } catch (error) {
    console.error('Registration error:', error);
    res.status(500).json({ error: 'Registration failed' });
  }
});

app.post('/api/verify', async (req, res) => {
  const { email, code } = req.body;
  try {
    const [users] = await pool.query('SELECT * FROM users WHERE email = ? AND verification_code = ?', [email, code]);
    if (users.length === 0) {
      return res.status(400).json({ error: 'Invalid email or verification code' });
    }

    const user = users[0];
    await pool.query('UPDATE users SET is_verified = TRUE, verification_code = NULL WHERE email = ?', [email]);

    req.session.userId = user.id;

    const [connections] = await pool.query('SELECT id FROM connections WHERE user_id = ?', [user.id]);
    let connectionId;
    if (connections.length === 0) {
      const [result] = await pool.query(
        'INSERT INTO connections (user_id, connection_name, status) VALUES (?, ?, ?)',
        [user.id, 'default_connection', 'disconnected']
      );
      connectionId = result.insertId;
    } else {
      connectionId = connections[0].id;
    }

    await pool.query(
      'INSERT INTO settings (id, user_id, connection_id, auto_reply_enabled, auto_reply_groups, afk_enabled, afk_message) VALUES (?, ?, ?, ?, ?, ?, ?) ON DUPLICATE KEY UPDATE auto_reply_enabled = ?, auto_reply_groups = ?, afk_enabled = ?, afk_message = ?',
      [user.id, user.id, connectionId, false, false, false, 'I am currently unavailable. Please leave a message.', false, false, false, 'I am currently unavailable. Please leave a message.']
    );

    res.json({ success: true, message: 'Account verified successfully' });
  } catch (error) {
    console.error('Verification error:', error);
    res.status(500).json({
      error: 'Verification failed',
      details: process.env.NODE_ENV === 'development' ? error.message : null
    });
  }
});

app.post('/api/login', async (req, res) => {
  const { email, password } = req.body;
  try {
    const [users] = await pool.query('SELECT * FROM users WHERE email = ?', [email]);
    if (users.length === 0) {
      return res.status(400).json({ error: 'Invalid credentials' });
    }

    const user = users[0];
    if (!user.is_verified) {
      return res.status(400).json({ error: 'Please verify your account first' });
    }

    const isValidPassword = await bcrypt.compare(password, user.password);
    if (!isValidPassword) {
      return res.status(400).json({ error: 'Invalid credentials' });
    }

    req.session.userId = user.id;
    res.json({ success: true, message: 'Login successful' });
  } catch (error) {
    console.error('Login error:', error);
    res.status(500).json({ error: 'Login failed' });
  }
});

app.post('/api/logout', (req, res) => {
  const userId = req.session.userId;
  req.session.destroy(async (err) => {
    if (err) {
      console.error('Logout error:', err);
      return res.status(500).json({ error: 'Logout failed' });
    }
    // Clean up WhatsApp connection
    if (userConnections.has(userId)) {
      try {
        await userConnections.get(userId).sock.logout();
        userConnections.get(userId).sock.ws.close();
      } catch (err) {
        console.error(`Error closing connection for user ${userId}:`, err);
      }
      userConnections.delete(userId);
    }
    res.json({ success: true, message: 'Logged out successfully' });
  });
});

// Other API endpoints
app.get('/api/user', isAuthenticated, async (req, res) => {
  try {
    const [users] = await pool.query('SELECT id, username, email FROM users WHERE id = ?', [req.session.userId]);
    if (users.length === 0) {
      return res.status(404).json({ error: 'User not found' });
    }
    res.json(users[0]);
  } catch (error) {
    console.error('User fetch error:', error);
    res.status(500).json({ error: 'Failed to fetch user' });
  }
});

app.get('/api/stats', isAuthenticated, async (req, res) => {
  try {
    const [rows] = await pool.query(`
      SELECT 
        SUM(CASE WHEN direction = 'incoming' THEN 1 ELSE 0 END) as received,
        SUM(CASE WHEN direction = 'outgoing' THEN 1 ELSE 0 END) as sent,
        MAX(created_at) as last_activity
      FROM messages
      WHERE user_id = ?
    `, [req.session.userId]);
    res.json(rows[0]);
  } catch (error) {
    console.error('Stats error:', error);
    res.status(500).json({ error: error.message });
  }
});

app.get('/api/messages', isAuthenticated, async (req, res) => {
  try {
    const [rows] = await pool.query(`
      SELECT * FROM messages 
      WHERE user_id = ? 
      ORDER BY created_at DESC LIMIT 50
    `, [req.session.userId]);
    res.json(rows);
  } catch (error) {
    console.error('Messages error:', error);
    res.status(500).json({ error: error.message });
  }
});

app.get('/api/settings', isAuthenticated, async (req, res) => {
  try {
    const [settings] = await pool.query(`
      SELECT * FROM settings 
      WHERE user_id = ?
    `, [req.session.userId]);
    const [autoReplies] = await pool.query(`
      SELECT * FROM auto_replies 
      WHERE user_id = ? 
      ORDER BY id
    `, [req.session.userId]);
    res.json({
      settings: settings[0] || {
        auto_reply_enabled: false,
        auto_reply_groups: false,
        afk_enabled: false,
        afk_message: 'I am currently unavailable. Please leave a message.'
      },
      auto_replies: autoReplies
    });
  } catch (error) {
    console.error('Settings error:', error);
    res.status(500).json({ error: error.message });
  }
});

app.post('/api/settings', isAuthenticated, async (req, res) => {
  const { auto_reply_enabled, auto_reply_groups, afk_enabled, afk_message } = req.body;
  try {
    const [connections] = await pool.query('SELECT id FROM connections WHERE user_id = ?', [req.session.userId]);
    if (connections.length === 0) {
      return res.status(400).json({ error: 'No connection found for user' });
    }
    const connectionId = connections[0].id;

    await pool.query(`
      INSERT INTO settings (id, user_id, connection_id, auto_reply_enabled, auto_reply_groups, afk_enabled, afk_message)
      VALUES (?, ?, ?, ?, ?, ?, ?)
      ON DUPLICATE KEY UPDATE
        auto_reply_enabled = ?,
        auto_reply_groups = ?,
        afk_enabled = ?,
        afk_message = ?
    `, [
      req.session.userId, req.session.userId, connectionId, 
      auto_reply_enabled, auto_reply_groups, afk_enabled, afk_message,
      auto_reply_enabled, auto_reply_groups, afk_enabled, afk_message
    ]);
    res.json({ success: true });
    io.to(`user-${req.session.userId}`).emit(`settings-updated-${req.session.userId}`, { auto_reply_enabled, auto_reply_groups, afk_enabled, afk_message });
  } catch (error) {
    console.error('Settings save error:', error);
    res.status(500).json({ error: error.message });
  }
});

app.get('/api/auto-replies', isAuthenticated, async (req, res) => {
  try {
    const [rows] = await pool.query(`
      SELECT * FROM auto_replies 
      WHERE user_id = ? 
      ORDER BY id
    `, [req.session.userId]);
    res.json(rows);
  } catch (error) {
    console.error('Auto-replies fetch error:', error);
    res.status(500).json({ error: error.message });
  }
});

app.post('/api/auto-replies', isAuthenticated, async (req, res) => {
  const { keyword, response } = req.body;
  console.log('Adding auto-reply:', { userId: req.session.userId, keyword, response });
  try {
    if (!keyword || !response) {
      console.log('Validation failed: Missing keyword or response');
      return res.status(400).json({ error: 'Keyword and response are required' });
    }
    if (keyword.length > 255) {
      console.log('Validation failed: Keyword too long');
      return res.status(400).json({ error: 'Keyword must be 255 characters or less' });
    }

    const [connections] = await pool.query('SELECT id FROM connections WHERE user_id = ? LIMIT 1', [req.session.userId]);
    if (connections.length === 0) {
      console.log('No connection found for user:', req.session.userId);
      return res.status(400).json({ error: 'No WhatsApp connection found for this user' });
    }
    const connectionId = connections[0].id;

    await pool.query(
      'INSERT INTO auto_replies (user_id, connection_id, keyword, response) VALUES (?, ?, ?, ?)',
      [req.session.userId, connectionId, keyword.toLowerCase(), response]
    );

    const [rows] = await pool.query('SELECT * FROM auto_replies WHERE user_id = ? ORDER BY id', [req.session.userId]);

    res.json({ success: true, auto_replies: rows });
    io.to(`user-${req.session.userId}`).emit(`auto-replies-updated-${req.session.userId}`, rows);
  } catch (error) {
    console.error('Auto-reply save error:', error.stack);
    res.status(500).json({ error: 'Failed to save auto-reply', details: error.message });
  }
});

app.delete('/api/auto-replies/:id', isAuthenticated, async (req, res) => {
  const { id } = req.params;
  try {
    await pool.query('DELETE FROM auto_replies WHERE id = ? AND user_id = ?', [id, req.session.userId]);
    const [rows] = await pool.query('SELECT * FROM auto_replies WHERE user_id = ? ORDER BY id', [req.session.userId]);
    res.json({ success: true, auto_replies: rows });
    io.to(`user-${req.session.userId}`).emit(`auto-replies-updated-${req.session.userId}`, rows);
  } catch (error) {
    console.error('Auto-reply delete error:', error);
    res.status(500).json({ error: error.message });
  }
});

app.get('/api/analytics/message-trends', isAuthenticated, async (req, res) => {
  try {
    const { period } = req.query;
    let interval, format;

    switch (period) {
      case '24h':
        interval = '1 HOUR';
        format = 'HH:mm';
        break;
      case '7d':
        interval = '1 DAY';
        format = 'MMM D';
        break;
      case '30d':
        interval = '1 DAY';
        format = 'MMM D';
        break;
      default:
        interval = '1 DAY';
        format = 'MMM D';
    }

    const [receivedData] = await pool.query(`
      SELECT 
        DATE_FORMAT(created_at, ?) as time,
        COUNT(*) as count
      FROM messages
      WHERE direction = 'incoming'
        AND user_id = ?
        AND created_at >= DATE_SUB(NOW(), INTERVAL ${period === '24h' ? '24 HOUR' : period === '7d' ? '7 DAY' : '30 DAY'})
      GROUP BY time
      ORDER BY created_at
    `, [format === 'HH:mm' ? '%H:00' : '%b %e', req.session.userId]);

    const [sentData] = await pool.query(`
      SELECT 
        DATE_FORMAT(created_at, ?) as time,
        COUNT(*) as count
      FROM messages
      WHERE direction = 'outgoing'
        AND user_id = ?
        AND created_at >= DATE_SUB(NOW(), INTERVAL ${period === '24h' ? '24 HOUR' : period === '7d' ? '7 DAY' : '30 DAY'})
      GROUP BY time
      ORDER BY created_at
    `, [format === 'HH:mm' ? '%H:00' : '%b %e', req.session.userId]);

    const now = moment();
    let timeLabels = [];
    let current = moment().subtract(period === '24h' ? 24 : period === '7d' ? 7 : 30, period === '24h' ? 'hours' : 'days');

    while (current <= now) {
      timeLabels.push(current.format(format));
      current.add(period === '24h' ? 1 : 1, period === '24h' ? 'hour' : 'day');
    }

    const receivedCounts = timeLabels.map(label => {
      const found = receivedData.find(item => item.time === label);
      return found ? found.count : 0;
    });

    const sentCounts = timeLabels.map(label => {
      const found = sentData.find(item => item.time === label);
      return found ? found.count : 0;
    });

    res.json({
      labels: timeLabels,
      received: receivedCounts,
      sent: sentCounts
    });
  } catch (error) {
    console.error('Message trends error:', error);
    res.status(500).json({ error: error.message });
  }
});

app.get('/api/analytics/top-contacts', isAuthenticated, async (req, res) => {
  try {
    const [rows] = await pool.query(`
      SELECT 
        sender,
        COUNT(*) as message_count,
        MAX(created_at) as last_active
      FROM messages
      WHERE direction = 'incoming'
        AND user_id = ?
      GROUP BY sender
      ORDER BY message_count DESC
      LIMIT 10
    `, [req.session.userId]);
    res.json(rows);
  } catch (error) {
    console.error('Top contacts error:', error);
    res.status(500).json({ error: error.message });
  }
});

app.get('/api/analytics/auto-reply-stats', isAuthenticated, async (req, res) => {
  try {
    const [triggerCount] = await pool.query(`
      SELECT COUNT(*) as total_triggers
      FROM messages m1
      WHERE m1.direction = 'outgoing'
        AND m1.user_id = ?
        AND EXISTS (
          SELECT 1 FROM messages m2
          WHERE m2.direction = 'incoming'
            AND m2.sender = m1.sender
            AND m2.user_id = ?
            AND m2.created_at < m1.created_at
            AND TIMESTAMPDIFF(MINUTE, m2.created_at, m1.created_at) <= 5
        )
    `, [req.session.userId, req.session.userId]);

    const [incomingCount] = await pool.query(`
      SELECT COUNT(*) as total_incoming
      FROM messages
      WHERE direction = 'incoming'
        AND user_id = ?
    `, [req.session.userId]);

    const [keywordUsage] = await pool.query(`
      SELECT 
        ar.keyword,
        COUNT(*) as count
      FROM messages m
      JOIN auto_replies ar ON m.message LIKE CONCAT('%', ar.keyword, '%')
      WHERE m.direction = 'incoming'
        AND m.user_id = ?
        AND ar.user_id = ?
      GROUP BY ar.keyword
      ORDER BY count DESC
      LIMIT 5
    `, [req.session.userId, req.session.userId]);

    const totalTriggers = triggerCount[0].total_triggers || 0;
    const totalIncoming = incomingCount[0].total_incoming || 1;
    const responseRate = Math.round((totalTriggers / totalIncoming) * 100);

    res.json({
      total_triggers: totalTriggers,
      response_rate: responseRate,
      top_keyword: keywordUsage.length > 0 ? keywordUsage[0].keyword : null,
      keyword_usage: keywordUsage
    });
  } catch (error) {
    console.error('Auto-reply stats error:', error);
    res.status(500).json({ error: error.message });
  }
});

app.get('/api/analytics/error-logs', isAuthenticated, async (req, res) => {
  try {
    const [rows] = await pool.query(`
      SELECT * FROM error_logs
      WHERE user_id = ?
      ORDER BY timestamp DESC
      LIMIT 20
    `, [req.session.userId]);
    res.json(rows);
  } catch (error) {
    console.error('Error logs error:', error);
    res.status(500).json({ error: error.message });
  }
});

// Updated routes for serving HTML files
app.get('/', (req, res) => {
  if (req.session.userId) {
    return res.redirect('/dashboard');
  }
  res.sendFile(path.join(__dirname, 'public', 'index.html'));
});

app.get('/login', (req, res) => {
  if (req.session.userId) {
    return res.redirect('/dashboard');
  }
  res.sendFile(path.join(__dirname, 'public', 'login.html'));
});

app.get('/register', (req, res) => {
  if (req.session.userId) {
    return res.redirect('/dashboard');
  }
  res.sendFile(path.join(__dirname, 'public', 'register.html'));
});

app.get('/verify', (req, res) => {
  if (req.session.userId) {
    return res.redirect('/dashboard');
  }
  res.sendFile(path.join(__dirname, 'public', 'verify.html'));
});

app.get('/dashboard', isAuthenticated, (req, res) => {
  res.sendFile(path.join(__dirname, 'public', 'dashboard.html'));
});

app.post('/api/generate-qr', isAuthenticated, async (req, res) => {
  const userId = req.session.userId;
  console.log(`Generating QR for user: ${userId}`);

  if (userConnections.has(userId)) {
    try {
      await userConnections.get(userId).sock.logout();
      userConnections.get(userId).sock.ws.close();
      userConnections.delete(userId);
    } catch (err) {
      console.error(`Error closing existing connection for user ${userId}:`, err);
    }
  }

  userConnections.set(userId, { sock: null, status: 'connecting', isGeneratingQR: true });
  io.to(`user-${userId}`).emit(`connection-status-${userId}`, { status: 'connecting' });

  try {
    await initWhatsAppBot(userId);
    res.json({ success: true, message: 'QR code generation initiated' });
  } catch (err) {
    console.error(`Failed to initialize WhatsApp bot for user ${userId}:`, err);
    userConnections.get(userId).status = 'disconnected';
    io.to(`user-${userId}`).emit(`connection-status-${userId}`, { status: 'disconnected' });
    res.status(500).json({ error: 'Failed to initialize WhatsApp connection' });
  }
});

app.post('/api/resend-code', async (req, res) => {
  const { email } = req.body;
  
  // Validate email format
  if (!email || !/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email)) {
    return res.status(400).json({ 
      success: false,
      error: 'Please provide a valid email address'
    });
  }

  try {
    const [users] = await pool.query('SELECT * FROM users WHERE email = ?', [email]);
    if (users.length === 0) {
      return res.status(400).json({ 
        success: false,
        error: 'Email not found. Please register first.'
      });
    }

    const user = users[0];
    if (user.is_verified) {
      return res.status(400).json({ 
        success: false,
        error: 'Account already verified. Please login.'
      });
    }

    const newCode = generateVerificationCode();
    await pool.query(
      'UPDATE users SET verification_code = ? WHERE email = ?',
      [newCode, email]
    );

    // Send email with new code
    try {
      await transporter.sendMail({
        from: '"WhatsApp Bot" <striker@joycloud.services>',
        to: email,
        subject: 'New Verification Code',
        html: `
          <h2>New Verification Code</h2>
          <p>Your new verification code is: <strong>${newCode}</strong></p>
          <p>Please enter this code to verify your account.</p>
          <p>If you didn't request this, please ignore this email.</p>
        `
      });

      console.log(`New verification code sent to ${email}`);
      return res.json({ success: true });
    } catch (emailError) {
      console.error('Email sending error:', emailError);
      return res.status(500).json({ 
        success: false,
        error: 'Failed to send verification email. Please try again later.'
      });
    }
  } catch (error) {
    console.error('Resend code error:', error);
    return res.status(500).json({ 
      success: false,
      error: 'Internal server error. Please try again.'
    });
  }
});

app.get('/api/connection-status', isAuthenticated, (req, res) => {
  const status = userConnections.get(req.session.userId)?.status || 'disconnected';
  res.json({ status });
});

io.on('connection', (socket) => {
  console.log('Client connected:', socket.id);

  socket.on('join-user', (userId) => {
    console.log(`User joined: ${userId}`);
    socket.join(`user-${userId}`);
    const status = userConnections.get(userId)?.status || 'disconnected';
    socket.emit(`connection-status-${userId}`, { status });
  });

  socket.on('disconnect', () => {
    console.log('Client disconnected:', socket.id);
  });
});

// Clustering for multi-core utilization
if (cluster.isMaster) {
  console.log(`Master ${process.pid} is running`);

  // Fork workers
  for (let i = 0; i < numCPUs; i++) {
    cluster.fork();
  }

  cluster.on('exit', (worker, code, signal) => {
    console.log(`Worker ${worker.process.pid} died`);
    cluster.fork(); // Restart worker
  });
} else {
  server.listen(3000, '0.0.0.0', async () => {
    console.log(`Worker ${process.pid} running on http://0.0.0.0:3000`);
    // Initialize database tables
    await pool.query(`
      CREATE TABLE IF NOT EXISTS users (
        id INT AUTO_INCREMENT PRIMARY KEY,
        username VARCHAR(255) NOT NULL,
        email VARCHAR(255) NOT NULL UNIQUE,
        password VARCHAR(255) NOT NULL,
        verification_code VARCHAR(6),
        is_verified BOOLEAN DEFAULT FALSE,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP
      )
    `);
    await pool.query(`
      CREATE TABLE IF NOT EXISTS sessions (
        session_id VARCHAR(128) PRIMARY KEY,
        expires INT UNSIGNED NOT NULL,
        data TEXT
      )
    `);
    await pool.query(`
      CREATE TABLE IF NOT EXISTS connections (
        id INT AUTO_INCREMENT PRIMARY KEY,
        user_id INT NOT NULL,
        connection_name VARCHAR(255) NOT NULL,
        status ENUM('connected', 'disconnected', 'connecting') DEFAULT 'disconnected',
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
      )
    `);
    await pool.query(`
      CREATE TABLE IF NOT EXISTS settings (
        id INT PRIMARY KEY,
        user_id INT NOT NULL,
        connection_id INT NOT NULL,
        auto_reply_enabled BOOLEAN DEFAULT FALSE,
        auto_reply_groups BOOLEAN DEFAULT FALSE,
        afk_enabled BOOLEAN DEFAULT FALSE,
        afk_message TEXT,
        FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
        FOREIGN KEY (connection_id) REFERENCES connections(id) ON DELETE CASCADE
      )
    `);
    await pool.query(`
      CREATE TABLE IF NOT EXISTS messages (
        id INT AUTO_INCREMENT PRIMARY KEY,
        user_id INT NOT NULL,
        connection_id INT NOT NULL,
        sender VARCHAR(255) NOT NULL,
        message TEXT NOT NULL,
        direction ENUM('incoming', 'outgoing') NOT NULL,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
        FOREIGN KEY (connection_id) REFERENCES connections(id) ON DELETE CASCADE
      )
    `);
    await pool.query(`
      CREATE TABLE IF NOT EXISTS auto_replies (
        id INT AUTO_INCREMENT PRIMARY KEY,
        user_id INT NOT NULL,
        connection_id INT NOT NULL,
        keyword VARCHAR(255) NOT NULL,
        response TEXT NOT NULL,
        FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
        FOREIGN KEY (connection_id) REFERENCES connections(id) ON DELETE CASCADE
      )
    `);
    await pool.query(`
      CREATE TABLE IF NOT EXISTS error_logs (
        id INT AUTO_INCREMENT PRIMARY KEY,
        user_id INT NOT NULL,
        connection_id INT NOT NULL,
        type VARCHAR(255) NOT NULL,
        message TEXT NOT NULL,
        timestamp DATETIME DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
        FOREIGN KEY (connection_id) REFERENCES connections(id) ON DELETE CASCADE
      )
    `);
  });
}